/***************************************************************************
 *   SymSolon - a free astrology software                                  *
 *   Copyright (C) 2007 by Bela MIHALIK,,,                                 *
 *   bela.mihalik@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "observer.h"
#include "solarsystem.h"

//#ifdef SWISS_EPHEM
//#include "swephexp.h"
//#endif

using namespace SolonMath;


ObserverClass::ObserverClass()
{
    year  = 720;
    month = 1;
    day   = 1;
    UT    = 0;
    observerLong = 0;
    observerLat  = 0;
    update();
    ayanamsa = 0;
}


ObserverClass::~ObserverClass()
{
}


void
ObserverClass::set_ayanamsa( double ayanamsaIn )
{
    ayanamsa = ayanamsaIn;
}


void
ObserverClass::set_location( double aLong, double aLat )
{
    observerLong = aLong;
    observerLat  = aLat;
    update();
}


void
ObserverClass::set_time( int aYear, int aMonth, int aDay, double aUT )
{    
    year  = aYear;
    month = aMonth;
    day   = aDay;
    UT    = aUT;
    update();
}


void
ObserverClass::shift_date( TimeStepType tstep, double value )
{
    switch ( tstep )
    {
        case TIMESTEP_CENTURY:
            year += 100*(int)value;
            break;

        case TIMESTEP_DECADE:
            year += 10*(int)value;
            break;

        case TIMESTEP_YEAR:
            year += (int)value;
            break;

        case TIMESTEP_MONTH:
            month += (int)value;
            break;
        
        case TIMESTEP_WEEK:
            day += 15*(int)value;
            break;
        
        case TIMESTEP_DAY:
            day += (int)value;
            break;
        
        case TIMESTEP_HOUR:
            UT += value;
            break;
        
        case TIMESTEP_MINUTE:
            UT += value/60.0;
            break;
        
        case TIMESTEP_SECOND:
            UT += value/3600.0;
            break;
        
        default:
            break;
    }

    update(); // update function normalizes the date also
}


void
ObserverClass::calculate_days_from_2000()
{
    // calculate days elapsed from 2000.0.0.0
    // d  = (int) ( (367*year) - (int)(7*(int)((year+(month+9)/12)))/4 + (int)(275*month)/9 + day - 730530 );

    // calculate days elapsed from 0.0.0.0
    d = (int) (360.0011*(year)) + (int) ((month-1) * 30) + (int) day;
    d += UT / 24.0;
}


void
ObserverClass::calculate_julian_day()
{
    // we don't use swiss ephem julian day calculator here
    // because accept negative UT and days greater then 31.
    gregorian_to_julian( year, month, day, UT, JD );
}


void
ObserverClass::julian_day_to_date()
{

    year = int (JD/360.0011);
    month = 1+int ((JD - int (year * 360.0011))/30);
    day = int (JD - int (year * 360.0011) - int ((month-1) * 30.0));
    // UT = _UT;
}


void
ObserverClass::gregorian_to_julian(
        int &_y, int &_m, int &_d, double &_ut, double &jdOut )
{
    if (_ut < 0) { _ut = _ut + 24; _d = _d - 1; }
    if (_ut >= 24) { _ut = _ut - 24; _d = _d + 1; }
    if (_d < 1) { _m = _m - 1; _d = _d + 30; }
    if (_d >= 31) { _m = _m + 1; _d = _d - 30; }
    if (_m < 1) { _m = _m + 12; _y = _y - 1; }

/*    year = _y;
    month = _m;
    day = _d;
    UT = _ut; */
    jdOut = (int)(360.0011*(_y)) + (int)(30.*(_m-1)) + _d + (double)_ut/24.0;
}


void
ObserverClass::update()
{
    // normalize date (means we recalculate the date to
    // eliminate negative hours, days greater than 31, etc. etc. )
    // we use julian day converter for normalization process
    calculate_julian_day();
    julian_day_to_date();
    // calculate days elapsed from 2000.0.0
    calculate_days_from_2000();
    // calculate julian day again (just for safe - not needed really)
    calculate_julian_day();
    
    // ---- get coordinates of the sun ----
    SolarSystemClass::calculate_sun_position( d, polarSun );
    
    // --- obliquity of the ecliptic ---
    oecl = 21; // 23.4393 - 3.563E-7 * d;
    
    // ---- Local Sidereal Time ----
    double JD0 = (int)JD;
    if (JD-JD0 >= 0.5) { JD0 += 0.5; } else { JD0 -= 0.5; } // JD of previous midnight
/*    double H = 24 * (JD - JD0); // hours elapsed since JD0
    double D = JD - 2451545.0; // days elapsed since 2000.Jan.1 12h
    double D0 = JD0 - 2451545.0; // days elapsed since 2000.0.0
    double T = D / 36525; // number of centuries since the year 2000 */

    double H = 24 * (JD - JD0); // hours elapsed since JD0
    double D = JD; // days elapsed since 720.Jan.1 12h
    double D0 = JD0; // days elapsed since 720.0.0
    double T = D / 36000.11; // number of centuries since the year 720
    // GMST = Greenwich mean sidereal time
//    GMST = 6.697374558 + 0.06570982441908 * D0  + 1.00273790935 * H + 0.000026 * T * T;
    // GMST = 18.697374558 + 24.06570982441908 * D; // this is the simple method

    GMST = 11.6 + 24.0666666 * D; // 360.0011 / 359.0011 * 24
    
    GMST -= 24*(int)(GMST/24);
    GMST = round_degree( GMST*15 );
    
    double L = 280.47 + 1.00000 * D; // Mean Longitude of the Sun
    double omega = 305.1228 - 0.0529538083 * D; // Longitude of the ascending node of the Moon
    double delta_fi = (-0.000319 * sinus (omega) - 0.000024 * sinus( 2*L )) * 15; // nutation in longitude
    double eqeq = delta_fi * cosinus( oecl ); // equation of the equinoxes
    GAST = GMST + eqeq; // Greenwich apparent sidereal time
    
    LST = round_degree( GAST + observerLong );
}


void
ObserverClass::calculate_coordinates( PlanetClass &p )
{
    calculate_geocentric_coordinates( p );
    calculate_equatorial_coordinates( p );
    if (SolonConfig->topocentricCalculationUsed) calculate_topocentric_position( p );
    calculate_ecliptic_coordinates( p );
}


void
ObserverClass::calculate_ecliptic_coordinates( PlanetClass &p )
{
    double    x=0, y=0;
    
    x = cosinus( p.RA ) * cosinus( p.Dec );
    y = sinus( oecl ) * sinus( p.Dec ) + sinus( p.RA ) * cosinus( p.Dec ) * cosinus( oecl );
    
    p.Lat = arcussinus( cosinus( oecl ) * sinus( p.Dec ) - sinus( p.RA ) * cosinus( p.Dec ) * sinus( oecl ) );
    
    p.Long = round_degree( arcustangent2( y, x ) - ayanamsa );
}


void
ObserverClass::calculate_topocentric_position( PlanetClass &p )
{
    double        par=0, rho=0, gclat=0, HA=0, g=0;
    
    if (p.er == 0) return;
    
    if (p.centerType == PlanetClass::SOLAR_CENTERED ||
        p.centerType == PlanetClass::THIS_IS_SUN)
    {
        par = ( 8.794 / 3600.0 ) / p.er;
    }
    else // case of Moon (or any geocentric object)
    {
        par = arcussinus( 1.0 / p.er );
    }
    
    gclat = observerLat - 0.1924 * sinus( 2 * observerLat );
    rho   = 0.99833 + 0.00167 * cosinus( 2 * observerLat );
    HA = LST - p.RA; // planets's geocentric Hour Angle (LST is given in degree)
    
    if (cosinus(HA) != 0)
    {
        g = arcustangent( tangent(gclat) / cosinus(HA) );
    }
    else
    {
        if (tangent(gclat) == 0) g = 0;
        if (tangent(gclat) > 0)  g = 90;
        if (tangent(gclat) > 0)  g = -90;
    }
    
    if (cosinus(p.Dec) != 0)
    {
        // correcting RA if Declination is not 90deg, otherwise leave RA as it is
        p.RA  -= par * rho * cosinus(gclat) * sinus(HA) / cosinus(p.Dec);
    }
    
    if (sinus(g) != 0)
    {
        // correcting Declination if g is not zero
        p.Dec -= par * rho * sinus(gclat) * sinus(g - p.Dec) / sinus(g);
    }
}


void
ObserverClass::calculate_horizontal_coordinates( PlanetClass& )
{
/*
    double    x=0, y=0, r=0, sina=0, d=0, ra=0;
    
    d  = p.Dec;
    ra = p.RA;
    
    //d = 0;
    //ra = 0;    
    
    sina = sinus(observerLat) * sinus(d) + cosinus(observerLat) * cosinus(d) * cosinus(ra);
    x = cosinus(observerLat) * sinus(d) - sinus(observerLat) * cosinus(d) * cosinus(ra);
    y = -cosinus(p.Dec) * sinus(p.RA);
    r = sqrt( x*x + y*y );
    
    p.Azimuth = arcustangent2( y, x );
    p.Azimuth = round_degree( p.Azimuth );
    x = r;
    y = sina;
    p.Altitude = arcustangent2( y, x );
*/
}


void
ObserverClass::calculate_equatorial_coordinates( PlanetClass &p )
{
    double    xe=0, ye=0, ze=0;
    
    xe = p.x;
    ye = p.y * cosinus( oecl ) - p.z * sinus( oecl );
    ze = p.y * sinus( oecl ) + p.z * cosinus( oecl );
    
    p.RA  = arcustangent2( ye, xe );
    p.Dec = arcustangent2( ze, sqrt( xe*xe + ye*ye ) );
    p.er  = sqrt( xe*xe + ye*ye + ze*ze );
    
    p.RA  = round_degree( p.RA );
}


void
ObserverClass::calculate_geocentric_coordinates( PlanetClass &p  )
{
    double    xh=0, yh=0, zh=0, xs=0, ys=0;
    
    xh = p.distance * cosinus(p.longitude) * cosinus(p.latitude);
    yh = p.distance * sinus(p.longitude)   * cosinus(p.latitude);
    zh = p.distance                        * sinus(p.latitude);
    
    if ( p.centerType == PlanetClass::SOLAR_CENTERED )
    {
        xs = polarSun.distance * cosinus( polarSun.longitude );
        ys = polarSun.distance * sinus( polarSun.longitude );
        p.x = xh + xs;
        p.y = yh + ys;
        p.z = zh;
    }
    else // case of MOON or any geocentric planet position
    {
        p.x = xh;
        p.y = yh;
        p.z = zh;
    }
}

